//
//  IconChooser.swift
//  Do It
//
//  Created by Jim Dovey on 2/6/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

struct IconChooser: View {
    @Binding var selectedIcon: String

    private struct IconSelectionInfo {
        let name: String
        let anchor: Anchor<CGRect>
    }

    private struct IconChoice: PreferenceKey {
        typealias Value = [IconSelectionInfo]
        static var defaultValue: Value = []
        static func reduce(value: inout [IconSelectionInfo],
                           nextValue: () -> [IconSelectionInfo]) {
            value.append(contentsOf: nextValue())
        }
    }

    private struct IconChoiceButtonStyle: ButtonStyle {
        func makeBody(configuration: Configuration) -> some View {
            configuration.label
                .font(.system(size: 24, weight: .bold, design: .rounded))
                .padding(6)
                .frame(width: 30)
                .padding(14)
                .background(Color(UIColor.tertiarySystemFill))
                .clipShape(Circle())
                .scaleEffect(configuration.isPressed ? 1.2 : 1)
        }
    }

    var body: some View {
        VStack(spacing: 14) {
            ForEach(listIconChoices, id: \.self) { rowData in
                HStack(spacing: 14) {
                    ForEach(rowData, id: \.self) { icon in
                        Button(action: { self.selectedIcon = icon }) {
                            Image(systemName: icon)
                        }
                        .accessibility(label: Text(LocalizedStringKey(icon)))
                        .accessibility(addTraits: self.selectedIcon == icon ? .isSelected : [])
                        .anchorPreference(key: IconChoice.self, value: .bounds) {
                            [IconSelectionInfo(name: icon, anchor: $0)]
                        }
                    }
                }
            }
        }
        .buttonStyle(IconChoiceButtonStyle())
        .backgroundPreferenceValue(IconChoice.self) { values in
            GeometryReader { proxy in
                self.selectionCircle(for: values, in: proxy)
            }
        }
    }

    private func selectionCircle(
        for prefs: [IconSelectionInfo],
        in proxy: GeometryProxy
    ) -> some View {
        let p = prefs.first { $0.name == selectedIcon }
        let bounds = p != nil ? proxy[p!.anchor] : .zero

        return Circle()
            .stroke(lineWidth: 3)
            .foregroundColor(Color(UIColor.separator))
            .frame(width: bounds.size.width + 12,
                   height: bounds.size.height + 12)
            .fixedSize()
            .offset(x: bounds.minX - 6, y: bounds.minY - 6)
    }
}

struct IconChooser_Previews: PreviewProvider {
    static var previews: some View {
        StatefulPreviewWrapper("list.bullet") {
            IconChooser(selectedIcon: $0)
        }
    }
}
